unit MailTemplateMessage;

{
  An XML message template handler for the mail template program.
  Requires MSXML v3 package from Microsoft.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Version 1.0 - October 17 1999.
  Version 1.1 - April 25, 2000.
}

interface

uses
  SysUtils, Classes, Windows, ActiveX, MSXML2_TLB, MailTemplateConst;

type
  TMailTemplateMessage = class(TObject)
  private
    FXMLDoc: IXMLDOMDocument;
  public
    constructor Create(Filename: string);
    destructor Destroy; override;
    function AttributeValue(NodeName, AttrName: string): string;
    function NodeValue(NodeName: string): string;
    function ParseMessage(Fields: TStrings): string;
  end;

implementation

resourcestring
  MissingField = 'Missing field: %s';
  NoMessage    = 'No message tag found in document';

{ TMailTemplateMessage ------------------------------------------------------------}

{ Initialisation }
constructor TMailTemplateMessage.Create(Filename: string);
begin
  inherited Create;
  FXMLDoc := CoDOMDocument.Create;
  FXMLDoc.PreserveWhiteSpace := True;
  FXMLDoc.Load(Filename);
end;

{ Release resources }
destructor TMailTemplateMessage.Destroy;
begin
  FXMLDoc := nil;
  inherited Destroy;
end;

{ Return the value of the named attribute -
  assumes only one such node }
function TMailTemplateMessage.AttributeValue(NodeName, AttrName: string): string;
var
  Elements: IXMLDOMNodeList;
begin
  Elements := FXMLDoc.GetElementsByTagName(NodeName);
  if Elements.Length = 0 then
    Result := ''
  else
    Result := Elements.Item[0].Attributes.GetNamedItem(AttrName).Text;
end;

{ Return the value of the named node -
  assumes only one such node and no children }
function TMailTemplateMessage.NodeValue(NodeName: string): string;
var
  Elements: IXMLDOMNodeList;
begin
  Elements := FXMLDoc.GetElementsByTagName(NodeName);
  if Elements.Length = 0 then
    Result := ''
  else
    Result := Elements.Item[0].Text;
end;

{ Parse the message tag and return its value }
function TMailTemplateMessage.ParseMessage(Fields: TStrings): string;
var
  Elements: IXMLDOMNodeList;
  FieldValue: string;
  Index: Integer;
begin
  Elements := FXMLDoc.GetElementsByTagName(MessageTag);
  if Elements.Length = 0 then
    raise EMailException.Create(NoMessage);

  Result := '';
  with Elements[0] do
    for Index := 0 to ChildNodes.Length - 1 do
      { Add text elements directly }
      if (ChildNodes[Index].NodeType = NODE_TEXT) or
          (ChildNodes[Index].NodeType = NODE_CDATA_SECTION) then
        Result := Result + ChildNodes[Index].Text
      { For 'field' elements get the field value }
      else if (ChildNodes[Index].NodeType = NODE_ELEMENT) and
          (ChildNodes[Index].NodeName = FieldTag) then
      begin
        FieldValue := Fields.Values[ChildNodes[Index].Text];
        if FieldValue = '' then
          { Error if no such field }
          raise EMailException.Create(
            Format(MissingField, [ChildNodes[Index].Text]));
        if FieldValue = Empty then
          { Replace empty field notation with empty string }
          FieldValue := '';
        Result := Result + FieldValue;
      end;
end;

initialization
  { Initialise COM }
  CoInitialize(nil);
finalization
  { Tidy up }
  CoUninitialize();
end.
